<?php
/*
Objective: This API fetches and inserts vendor records in the `vendor_header_all` table.
Also handles quotation file uploads for a specific requisition.
File Version: 1.0
API No: KDN11
Date of Creation: 01/04/2025
Last Update: 02/04/2025

Logic Zone:

Request Method:
Type: GET (for fetching vendors) / POST (for inserting a new vendor and uploading quotation files)

Headers: Content-Type: application/json

Request Parameters (for POST):
- vendor_name: Name of the vendor (string)
- vendor_address: Address of the vendor (string)
- vendor_currency: Currency of the vendor (string)
- quotation_files: Array of objects containing ven_id, file_name, amount, currency, and requisition_no.

Process:

* GET Request: Fetch Vendor Records
  The API queries the `vendor_header_all` table to fetch the vendor ID (`ven_id`) and vendor name (`ven_name`) for all vendors.

* POST Request: Insert New Vendor or Upload Quotation Files
  The API inserts a new vendor into the `vendor_header_all` table or uploads quotation file data into `requisition_header_all`.

Response Structure (for GET Request):
* Success Response:
  If the vendor records are successfully fetched, the API returns:
    - success_code: 200 (indicating success).
    - message: "Vendors fetched successfully".
    - vendors: Array of vendor records including vendor ID and vendor name.

* Failure Response:
  If there is an issue with the query execution, the response contains:
    - error_code: 500 (Internal Server Error).
    - message: "Failed to fetch vendor records".

Response Structure (for POST Request):
* Success Response:
  If the vendor is successfully inserted or quotation files are uploaded, the API returns:
    - success_code: 200 (indicating success).
    - message: "Vendor added successfully" or "Quotation files uploaded successfully".

* Failure Response:
  If there is an issue with the insertion or uploading, the response contains:
    - error_code: 500 (Internal Server Error).
    - message: "Failed to add vendor" or "Failed to upload quotation files".

Error Zone:
*/

session_start();
require_once("../config/config.php");

$conn = connection::getConnection();

// Fetch Vendor Records
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $sql = "SELECT ven_id, ven_name, ven_currency FROM vendor_header_all";
    $result = $conn->query($sql);

    if ($result) {
        $vendors = [];
        while ($row = $result->fetch_assoc()) {
            $vendors[] = [
                'vendor_id' => $row['ven_id'],
                'vendor_name' => $row['ven_name'],
                'vendor_currency' => $row['ven_currency']
            ];
        }

        if (!empty($vendors)) {
            $response = [
                'success_code' => 200,
                'message' => 'Vendors fetched successfully',
                'vendors' => $vendors
            ];
            echo json_encode($response);
        } else {
            $response = [
                'error_code' => 404,
                'message' => 'No vendors found'
            ];
            echo json_encode($response);
        }
    } else {
        log_error("Query failed: " . $conn->error);
        $response = [
            'error_code' => 500,
            'message' => 'Failed to fetch vendor records'
        ];
        echo json_encode($response);
    }
    exit(); 
}

// Insert New Vendor and Upload Quotation Files 
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $data = json_decode(file_get_contents("php://input"), true);

    // Insert New Vendor
    if (isset($data['vendor_name']) && isset($data['vendor_address']) && isset($data['vendor_currency'])) {
        $vendorName = $conn->real_escape_string($data['vendor_name']);
        $vendorAddress = $conn->real_escape_string($data['vendor_address']);
        $vendorCurrency = $conn->real_escape_string($data['vendor_currency']);

        // Check if the vendor already exists
        $checkVendorSql = "SELECT ven_id FROM vendor_header_all WHERE ven_name = '$vendorName'";
        $checkVendorResult = $conn->query($checkVendorSql);

        if ($checkVendorResult && $checkVendorResult->num_rows > 0) {
            $response = [
                'error_code' => 400,
                'message' => 'Vendor already exists.'
            ];
            echo json_encode($response);
            exit();
        }

        // insert new vendor in database
        $sql = "INSERT INTO vendor_header_all (ven_name, ven_address, ven_currency, ven_status, ven_date_of_creation) 
                VALUES ('$vendorName', '$vendorAddress', '$vendorCurrency', 1, NOW())"; 

        if ($conn->query($sql)) {
            $response = [
                'success_code' => 200,
                'message' => 'Vendor added successfully.'
            ];
            echo json_encode($response);
            exit(); 
        } else {
            error_log("SQL Error: " . $conn->error);
            $response = [
                'error_code' => 500, 
                'message' => 'Failed to add vendor. Please check the logs for more details.'
            ];
            echo json_encode($response);
            exit();
        }
    }


    // Upload Quotation Files
    if (isset($data['quotation_files']) && isset($data['requisition_no'])) {
        $quotationFiles = $data['quotation_files'];
        $requisitionNo = $conn->real_escape_string($data['requisition_no']);
    
        $allUploaded = true;
        $failedFiles = [];
    
        // Process each quotation file
        foreach ($quotationFiles as $quotation) {
            $vendorId = (int)$quotation['ven_id'];
            $fileName = $conn->real_escape_string($quotation['file_name']);
            $amount = (float)$quotation['amount'];
            $currency = $conn->real_escape_string($quotation['currency']);
    
            $sql = "SELECT req_quotation_files FROM requisition_header_all WHERE req_no = '$requisitionNo'";
            $result = $conn->query($sql);
    
            if ($result) {
                $row = $result->fetch_assoc();
                $existingQuotationFiles = $row['req_quotation_files'];
    
                // Decode the existing JSON into a PHP array
                if (!empty($existingQuotationFiles)) {
                    $existingQuotationFilesArray = json_decode($existingQuotationFiles, true);
                } else {
                    $existingQuotationFilesArray = []; 
                }
    
                $newQuotation = [
                    'ven_id' => $vendorId,
                    'file_name' => $fileName,
                    'amount' => $amount,
                    'currency' => $currency
                ];
    
                $existingQuotationFilesArray[] = $newQuotation;
                $updatedQuotationFilesJson = json_encode($existingQuotationFilesArray);
    
                // Update the requisition_header_all table with the new JSON data
                $updateSql = "UPDATE requisition_header_all 
                              SET req_quotation_files = '$updatedQuotationFilesJson' 
                              WHERE req_no = '$requisitionNo'";
    
                if (!$conn->query($updateSql)) {
                    $allUploaded = false;
                    $failedFiles[] = $quotation;
                    error_log("Failed to execute query for requisition_no: " . $requisitionNo);
                }
            } else {
                $allUploaded = false;
                error_log("Failed to fetch existing quotation files for requisition_no: " . $requisitionNo);
            }
        }
    
        // Return response
        if ($allUploaded) {
            $response = [
                'success_code' => 200,
                'message' => 'Quotation files uploaded successfully.'
            ];
        } else {
            $response = [
                'error_code' => 500,
                'message' => 'Failed to upload some quotation files.',
                'failed_files' => $failedFiles
            ];
        }
    
        echo json_encode($response);
        exit(); 
    }
}

$conn->close();
?>
