<?php

/*
objective : This API handles the addition and retrieval of equipment within specific categories and departments. 
It supports adding new equipment to a department/category and fetching equipment details with pagination and optional search functionality.

file version : 1.0
API NO : KDN04
Date of creation : 24/03/2025

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Logic Zone : 

Date  : 24/03/2025

Request Method
Type: POST, GET

Headers: Content-Type: application/json

Request Parameters
- For POST:
  - dept_id (int, required) – The ID of the department.
  - eq_name (string, required) – The name of the equipment.
  - eq_type (string, required) – The type of the equipment.
  - model_name (string, optional) – The model name of the equipment.
  - maker_name (string, optional) – The name of the equipment maker.
  - maker_address (string, optional) – The address of the equipment maker.
  - cat_id (int, optional) – The category ID of the equipment (if available).

- For GET:
  - action (string, optional) – Specifies action type ('fetch_departments' to fetch department details, 'fetch_categories' to fetch categories for a department).
  - dept_id (int, optional) – The department ID for which equipment is to be retrieved.
  - search (string, optional) – Search query for filtering equipment names.
  - page (int, optional) – The page number for pagination. Default is 1.

Authentication: No authentication is required for this API.

***********

Authentication Process

* Step 1: Input Validation
  - For POST, it checks if department ID, equipment name, and equipment type are provided.
  - For GET, it checks if department ID is provided (if applicable), and applies pagination with search.

* Step 2: Duplicate Check (POST only)
  - It checks if the equipment already exists in the specified department and category.
  - If the equipment exists, an error response is returned.

* Step 3: Inserting Equipment (POST only)
  - If no duplicate is found, a new equipment entry is inserted into the `equipment_header_all` table.
  - If the equipment insertion is successful, the maker's details are also inserted into the `modal_maker_header_all` table.

* Step 4: Fetching Equipment Data (GET only)
  - For GET requests, it returns a paginated list of equipment, optionally filtered by department and search query.
  - It calculates the total number of pages for pagination based on the number of matching equipment.

***********

Response Structure

* Success Response
  - For POST:
    - error_code: 200 (indicating success)
    - message: 'Equipment added successfully'

  - For GET:
    - error_code: 200 (indicating success)
    - data: Array of equipment details
    - total_pages: Total number of pages for pagination
    - current_page: The current page number

* Failure Responses:
  - For POST:
    - error_code: 100 (Equipment already exists in the department and category)
    - error_code: 400 (Missing required department ID, equipment name, or equipment type)
    - error_code: 102 (Error while adding equipment or maker details)

  - For GET:
    - error_code: 101 (No equipment found)
    - error_code: 300 (Invalid request method)

Date :
xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Error Zone : 

Date : 
*/

session_start();
require_once("../config/config.php");

$itemsPerPage = 5;

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['dept_id']) && isset($_POST['eq_name']) && isset($_POST['eq_type']) && isset($_POST['model_name']) && isset($_POST['maker_name']) && isset($_POST['maker_address']) && !empty($_POST['dept_id']) && !empty($_POST['eq_name']) && !empty($_POST['eq_type'])) {
        $conn = connection::getConnection();

        // Sanitize input to prevent SQL injection
        $dept_id = mysqli_real_escape_string($conn, $_POST['dept_id']);
        $cat_id = isset($_POST['cat_id']) ? mysqli_real_escape_string($conn, $_POST['cat_id']) : NULL;
        $eq_name = mysqli_real_escape_string($conn, $_POST['eq_name']);
        $eq_type = mysqli_real_escape_string($conn, $_POST['eq_type']);
        $model_name = mysqli_real_escape_string($conn, $_POST['model_name']);
        $maker_name = mysqli_real_escape_string($conn, $_POST['maker_name']);
        $maker_address = mysqli_real_escape_string($conn, $_POST['maker_address']);
        $date_of_creation = date('Y-m-d H:i:s');

        // Check if equipment already exists in the same department and category
        $checkQuery = "SELECT eq_name FROM equipment_header_all WHERE dept_id = '$dept_id' AND cat_id = '$cat_id' AND eq_name = '$eq_name'";
        $checkResult = mysqli_query($conn, $checkQuery);
        
        if (mysqli_num_rows($checkResult) > 0) {
            echo json_encode(['success' => false, 'message' => 'This equipment already exists in the selected department and category.']);
        } else {
            // Insert Equipment into equipment_header_all table
            $sql = "INSERT INTO equipment_header_all (dept_id, cat_id, eq_name, eq_engine_type, eq_date_of_creation) 
                    VALUES ('$dept_id', '$cat_id', '$eq_name', '$eq_type', '$date_of_creation')";

            if (mysqli_query($conn, $sql)) {
                $eq_id = mysqli_insert_id($conn);  // Get the last inserted equipment id

                // Insert into modal_maker_header_all table
                $maker_sql = "INSERT INTO modal_maker_header_all (dept_id, cat_id, eq_id, model_name, maker_name, maker_address, item_date_of_creation) 
                              VALUES ('$dept_id', '$cat_id', '$eq_id', '$model_name', '$maker_name', '$maker_address', '$date_of_creation')";

                if (mysqli_query($conn, $maker_sql)) {
                    echo json_encode(['success' => true, 'message' => 'Equipment added successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Error adding maker details: ' . mysqli_error($conn)]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Error adding equipment: ' . mysqli_error($conn)]);
            }
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'All fields are required']);
    }
} elseif ($_SERVER['REQUEST_METHOD'] == 'GET') {
    $conn = connection::getConnection();
    if (isset($_GET['action']) && $_GET['action'] == 'fetch_departments') {
        $sql = "SELECT dept_id, dept_name FROM department_header_all";
        $result = mysqli_query($conn, $sql);

        $departments = [];
        while ($row = mysqli_fetch_assoc($result)) {
            $departments[] = $row;
        }
        echo json_encode($departments);
    } elseif (isset($_GET['action']) && $_GET['action'] == 'fetch_categories') {
        $dept_id = isset($_GET['dept_id']) ? (int)$_GET['dept_id'] : 0;
        $sql = "SELECT cat_id, cat_name FROM category_header_all WHERE dept_id = '$dept_id'";
        $result = mysqli_query($conn, $sql);

        $categories = [];
        while ($row = mysqli_fetch_assoc($result)) {
            $categories[] = $row;
        }
        echo json_encode($categories);
    } else {
        $search = isset($_GET['search']) ? mysqli_real_escape_string($conn, $_GET['search']) : '';
        $dept_id = isset($_GET['dept_id']) ? (int)$_GET['dept_id'] : 0;
        $currentPage = isset($_GET['page']) ? (int)$_GET['page'] : 1;

        // Query to get total equipment count for pagination with search
        $totalEquipmentQuery = "SELECT COUNT(*) AS total FROM equipment_header_all WHERE eq_name LIKE '%$search%'";
        if ($dept_id) {
            $totalEquipmentQuery .= " AND dept_id = '$dept_id'";
        }
        $totalEquipmentResult = mysqli_query($conn, $totalEquipmentQuery);
        $totalEquipment = mysqli_fetch_assoc($totalEquipmentResult)['total'];
        $totalPages = ceil($totalEquipment / $itemsPerPage);
        $currentPage = max(1, min($currentPage, $totalPages));
        $start = ($currentPage - 1) * $itemsPerPage;

        // Query to fetch equipment for the current page with search
        $sql = "SELECT e.eq_id, e.eq_name, e.eq_engine_type, m.model_name, m.maker_name, d.dept_name, c.cat_name 
                FROM equipment_header_all e 
                LEFT JOIN modal_maker_header_all m ON e.eq_id = m.eq_id
                LEFT JOIN department_header_all d ON e.dept_id = d.dept_id
                LEFT JOIN category_header_all c ON e.cat_id = c.cat_id
                WHERE e.eq_name LIKE '%$search%'";

        if ($dept_id) {
            $sql .= " AND e.dept_id = '$dept_id'";
        }

        $sql .= " LIMIT $start, $itemsPerPage";

        $result = mysqli_query($conn, $sql);

        $equipments = [];
        if ($result && mysqli_num_rows($result) > 0) {
            while ($row = mysqli_fetch_assoc($result)) {
                $equipments[] = $row;
            }
            echo json_encode([
                'success' => true,
                'data' => $equipments,
                'total_pages' => $totalPages,
                'current_page' => $currentPage
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'No equipment found']);
        }
    }
} else {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
}
?>
