<?php
/*
Objective: This API retrieves accepted requisition records (status = 2) from the database with pagination, including related ship and department details. 
           If a requisition number is provided, it fetches detailed information for that requisition.
File Version: 1.0
API No: KDN10
Date of Creation: 01/04/2025
Last Update: 01/04/2025

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Logic Zone:

Date: 01/04/2025

Request Method:
Type: POST or GET (depending on the request)

Headers: Content-Type: application/json

Request Parameters:
The API expects the following parameters in the request body:
- page (int, optional) – The page number for pagination. If not provided, defaults to page 1.
- requisition_no (GET parameter) – The requisition number for fetching detailed requisition information.

Process:

* Step 1: Input Validation
  The system checks if the `page` parameter is provided. If it's missing, the API defaults to page 1.

* Step 2: Fetch Requisition Records
  If no `requisition_no` is provided, the API queries the database to fetch accepted requisition records for the specified page, 
  with a limit of 6 records per page. It also joins the `ship_header_all` and `department_header_all` tables to fetch the related ship and department names.

* Step 3: Fetch Detailed Requisition Information
  If a `requisition_no` is provided, the API queries the database to fetch detailed information for that requisition, 
  including ship, department, equipment, model, and category information.

* Step 4: Pagination and Total Records Count
  The total number of requisition records is calculated using `COUNT(*)` to determine the total number of pages required for pagination.

* Step 5: Return Response
  The API returns a response with requisition records, the total number of pages, and a success message. 
  If a requisition number is provided, the API returns detailed requisition information.

Response Structure:

* Success Response:
  If the requisition records are successfully fetched, the API returns:
    - success_code: 200 (indicating success).
    - message: "Accepted requisitions fetched successfully".
    - requests: Array of accepted requisition details including requisition number, ship name, department name, requested date, and status.
    - totalPages: The total number of pages available based on the records and pagination limit.
    - data: Detailed requisition information (if a requisition number is provided).

* Failure Response:
  If there is an issue with the query execution, the response contains:
    - error_code: 500 (Internal Server Error).
    - message: "Failed to fetch accepted requisition records".

Date:

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Error Zone:

Date:
*/

session_start();
require_once("../config/config.php");

$conn = SqlServerConnection::getConnection();

// Check if req_id is provided (for fetching detailed requisition info)
if (isset($_GET['req_id'])) {
    $req_id = $_GET['req_id'];

    // Query to fetch requisition header details with related data
    $sql = "SELECT r.req_no, r.req_date_of_creation, r.req_priority, r.req_remark, r.req_view, r.ship_id, r.dept_id, r.cat_id, r.eq_id, r.m_id, 
            (SELECT ship_name FROM ship_header_all WHERE ship_id = r.ship_id) AS ship_name, 
            (SELECT dept_name FROM department_header_all WHERE dept_id = r.dept_id) AS dept_name, 
            (SELECT eq_name FROM equipment_header_all WHERE eq_id = r.eq_id) AS eq_name, 
            (SELECT model_name FROM modal_maker_header_all WHERE m_id = r.m_id) AS model_name, 
            (SELECT maker_name FROM modal_maker_header_all WHERE m_id = r.m_id) AS maker_name, 
            (SELECT maker_address FROM modal_maker_header_all WHERE m_id = r.m_id) AS maker_address, 
            (SELECT cat_name FROM category_header_all WHERE cat_id = r.cat_id) AS cat_name 
            FROM requisition_header_all r WHERE r.req_id = ? AND r.req_status = 2";
    $params = [$req_id];
    $stmt = sqlsrv_query($conn, $sql, $params);

    if ($stmt && sqlsrv_has_rows($stmt)) {
        $requisition_details = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC);
        $priority_level = ($requisition_details['req_priority'] == 1) ? 'Normal' : 'Urgent';
        $creation_date = $requisition_details['req_date_of_creation']->format('d/m/Y');

        // Get requisition items
        $query_items = "SELECT * FROM item_requisition_header_all WHERE req_id = ?";
        $stmt_items = sqlsrv_query($conn, $query_items, [$req_id]);

        $items_data = [];
        while ($row = sqlsrv_fetch_array($stmt_items, SQLSRV_FETCH_ASSOC)) {
            $sub_cat_id = $row['sub_cat_id'];
            $sub_cat_name = "";

            $sub_cat_stmt = sqlsrv_query($conn, "SELECT sub_cat_name FROM sub_category_header_all WHERE sub_cat_id = ?", [$sub_cat_id]);
            if ($sub_cat_stmt) {
                $row_sub = sqlsrv_fetch_array($sub_cat_stmt, SQLSRV_FETCH_ASSOC);
                $sub_cat_name = $row_sub['sub_cat_name'] ?? '';
            }

            if (!isset($items_data[$sub_cat_id])) {
                $items_data[$sub_cat_id] = [
                    "sub_cat_id" => $sub_cat_id,
                    "sub_cat_name" => $sub_cat_name,
                    "items" => []
                ];
            }

            $items = [];
            for ($i = 1; $i <= 40; $i++) {
                if (!empty($row["item_id_$i"])) {
                    $item_details = json_decode($row["item_details_$i"]);
                    if (!$item_details) continue;
                    $last_item = end($item_details);
                    $item_id = $last_item->item_id;

                    if ($last_item->status != 2) {
                        $item_stmt = sqlsrv_query($conn, "SELECT item_id, item_name, item_drawing_no, item_part_no 
                            FROM item_header_all WHERE item_id = ?", [$item_id]);
                        $item_row = sqlsrv_fetch_array($item_stmt, SQLSRV_FETCH_ASSOC);

                        $item_details_for_particular_item = [
                            "item_id" => $last_item->item_id,
                            "item_name" => $item_row['item_name'],
                            "item_drawing_no" => $item_row['item_drawing_no'],
                            "item_part_no" => $item_row['item_part_no'],
                            "working_qty" => $last_item->working_qty,
                            "rob" => $last_item->rob,
                            "req_qty" => $last_item->req_qty,
                            "accepted_qty" => $last_item->accepted_qty,
                            "pending_qty" => $last_item->pending_qty ?? 0,
                            "remark" => $last_item->remark,
                            "status" => $last_item->status
                        ];

                        // Fetch vendor details
                        $vendor_stmt = sqlsrv_query($conn, "SELECT * FROM vendor_item_allocation_all WHERE item_id = ?", [$item_id]);
                        $last_vendor_details = [];

                        while ($vendor_row = sqlsrv_fetch_array($vendor_stmt, SQLSRV_FETCH_ASSOC)) {
                            for ($p = 1; $p <= 30; $p++) {
                                if (!empty($vendor_row["ven_id_$p"])) {
                                    $ven_id = $vendor_row["ven_id_$p"];
                                    $ven_detail_json = $vendor_row["ven_id_details_$p"];
                                    $ven_detail = json_decode($ven_detail_json, true);

                                    $ven_stmt = sqlsrv_query($conn, "SELECT * FROM vendor_header_all WHERE ven_id = ?", [$ven_id]);
                                    $ven_data = sqlsrv_fetch_array($ven_stmt, SQLSRV_FETCH_ASSOC);

                                    $last_vendor_details[] = [
                                        'ven_name' => $ven_data['ven_name'],
                                        'avg' => $ven_detail['avg'],
                                        'last_purches_amt' => $ven_detail['last_purches_amt'],
                                        'currency' => $ven_data['ven_currency'],
                                        'last_purches_date' => $ven_detail['last_purches_date']
                                    ];
                                } else {
                                    break;
                                }
                            }
                        }

                        $items[] = [
                            "item_id" => $row["item_id_$i"],
                            "item_details" => $item_details_for_particular_item,
                            "vendor_details" => $last_vendor_details
                        ];
                    }
                }
            }

            $items_data[$sub_cat_id]["items"] = array_merge($items_data[$sub_cat_id]["items"], $items);
        }

        $items_data = array_values($items_data);

        $response = [
            'status' => 'success',
            'data' => [
                'req_id' => $req_id,
                'requisition_no' => $requisition_details['req_no'],
                'creation_date' => $creation_date,
                'priority_level' => $priority_level,
                'ship_name' => $requisition_details['ship_name'],
                'dept_name' => $requisition_details['dept_name'],
                'eq_name' => $requisition_details['eq_name'],
                'model_name' => $requisition_details['model_name'],
                'maker_name' => $requisition_details['maker_name'],
                'maker_address' => $requisition_details['maker_address'],
                'cat_name' => $requisition_details['cat_name'],
                'items' => $items_data
            ]
        ];
        echo json_encode($response);

    } else {
        echo json_encode([
            'status' => 'error',
            'message' => 'No requisition found with the given req_id.'
        ]);
    }

} else {
    // Pagination for list of requisitions
    $page = isset($_POST['page']) ? (int)$_POST['page'] : 1;
    $limit = 6;
    $offset = ($page - 1) * $limit;

    $sql = "SELECT r.req_id, r.req_no, r.ship_id, sha.ship_name, r.dept_id, dha.dept_name, r.req_status, r.req_date_of_creation 
            FROM requisition_header_all r 
            LEFT JOIN ship_header_all sha ON r.ship_id = sha.ship_id 
            LEFT JOIN department_header_all dha ON r.dept_id = dha.dept_id 
            WHERE r.req_status = 2
            ORDER BY r.req_id DESC 
            OFFSET ? ROWS FETCH NEXT ? ROWS ONLY";
    $stmt = sqlsrv_query($conn, $sql, [$offset, $limit]);

    if ($stmt) {
        $requests = [];
        while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
            $requests[] = [
                'req_id' => $row['req_id'],
                'requisition_no' => $row['req_no'],
                'ship_name' => $row['ship_name'],
                'department' => $row['dept_name'],
                'requested_date' => $row['req_date_of_creation']->format('Y-m-d'),
                'status' => $row['req_status'],
            ];
        }

        $sqlCount = "SELECT COUNT(*) AS total FROM requisition_header_all WHERE req_status = 2";
        $stmtCount = sqlsrv_query($conn, $sqlCount);
        $rowCount = sqlsrv_fetch_array($stmtCount, SQLSRV_FETCH_ASSOC);
        $totalRecords = $rowCount['total'];
        $totalPages = ceil($totalRecords / $limit);

        $response = [
            'success_code' => 200,
            'message' => 'Accepted requisitions fetched successfully',
            'requests' => $requests,
            'totalPages' => $totalPages
        ];
        echo json_encode($response);
    } else {
        echo json_encode([
            'error_code' => 500,
            'message' => 'Failed to fetch accepted requisition records'
        ]);
    }
}

SqlServerConnection::closeConnection();
?>
