<?php
/*
Objective: This API retrieves requisition records from the database with pagination, including related ship and department details. 
           If a requisition number is provided, it fetches detailed information for that requisition.
File Version: 1.0
API No: KDN01
Date of Creation: 25/03/2015
Last Update: 31/03/2025

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Logic Zone:

Date: 25/03/2015

Request Method:
Type: POST or GET (depending on the request)

Headers: Content-Type: application/json

Request Parameters:
The API expects the following parameters in the request body:
- page (int, optional) – The page number for pagination. If not provided, defaults to page 1.
- requisition_no (GET parameter) – The requisition number for fetching detailed requisition information.

Process:

* Step 1: Input Validation
  The system checks if the `page` parameter is provided. If it's missing, the API defaults to page 1.

* Step 2: Fetch Requisition Records
  If no `requisition_no` is provided, the API queries the database to fetch requisition records for the specified page, 
  with a limit of 6 records per page. It also joins the `ship_header_all` and `department_header_all` tables to fetch the related ship and department names.

* Step 3: Fetch Detailed Requisition Information
  If a `requisition_no` is provided, the API queries the database to fetch detailed information for that requisition, 
  including ship, department, equipment, model, and category information.

* Step 4: Pagination and Total Records Count
  The total number of requisition records is calculated using `COUNT(*)` to determine the total number of pages required for pagination.

* Step 5: Return Response
  The API returns a response with requisition records, the total number of pages, and a success message. 
  If a requisition number is provided, the API returns detailed requisition information.

Response Structure:

* Success Response:
  If the requisition records are successfully fetched, the API returns:
    - success_code: 200 (indicating success).
    - message: "Requisitions fetched successfully".
    - requests: Array of requisition details including requisition number, ship name, department name, requested date, and status.
    - totalPages: The total number of pages available based on the records and pagination limit.
    - data: Detailed requisition information (if a requisition number is provided).

* Failure Response:
  If there is an issue with the query execution, the response contains:
    - error_code: 500 (Internal Server Error).
    - message: "Failed to fetch requisition records".

Date:

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Error Zone:

Date:
*/

session_start();
require_once("../config/config.php");

$conn = SqlServerConnection::getConnection();

$action = isset($_POST['action']) ? $_POST['action'] : '';
$requisitionId = isset($_POST['requisition_id']) ? $_POST['requisition_id'] : ''; 

if ($action && $requisitionId) {
    $statusMap = ['accept' => 2, 'reject' => 3];

    if (isset($statusMap[$action])) {
        $status = $statusMap[$action];

        sqlsrv_begin_transaction($conn);

        $updateRequisitionSql = "UPDATE requisition_header_all SET req_status = ?, req_view = 1 WHERE req_id = ?";
        $requisitionStmt = sqlsrv_query($conn, $updateRequisitionSql, [$status, $requisitionId]);

        if ($requisitionStmt) {
            $updateTrackerSql = "UPDATE tracker_header_all SET order_placed = GETDATE() WHERE req_id = ?";
            $trackerStmt = sqlsrv_query($conn, $updateTrackerSql, [$requisitionId]);

            if ($trackerStmt) {
                sqlsrv_commit($conn);
                echo json_encode(['success' => true]);
            } else {
                sqlsrv_rollback($conn);
                echo json_encode(['success' => false, 'error' => 'Failed to update tracker.']);
            }
        } else {
            sqlsrv_rollback($conn);
            echo json_encode(['success' => false, 'error' => 'Failed to update requisition.']);
        }
    } else {
        echo json_encode(['success' => false, 'error' => 'Invalid action']);
    }
} else {
    if (isset($_GET['requisition_id'])) {  
        $requisition_id = $_GET['requisition_id']; 

        // Fetch detailed requisition information based on the new schema using req_id
        $sql = "SELECT r.req_id, r.req_no, r.ship_id, r.dept_id, r.req_serial_no,
                       r.req_for, r.req_priority, r.req_status, r.req_eq_name_engine_type, r.req_model_type, r.req_maker_name, 
                       r.req_maker_address, r.req_remark, r.req_date_of_creation, r.user_id, r.req_view, r.req_edit_count, 
                       r.req_edit_details, r.req_quotation_files, 
                       (SELECT ship_name FROM ship_header_all WHERE ship_id = r.ship_id) AS ship_name, 
                       (SELECT dept_name FROM department_header_all WHERE dept_id = r.dept_id) AS dept_name 
                FROM requisition_header_all r 
                WHERE r.req_id = ?";  
        $stmt = sqlsrv_query($conn, $sql, [$requisition_id]);

        if ($stmt && ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC))) {
            $priority_level = ($row['req_priority'] == 1) ? 'Normal' : 'Urgent';
            $creation_date = $row['req_date_of_creation'] instanceof DateTime
                ? $row['req_date_of_creation']->format('d/m/Y')
                : '';

            // Fetch item requisition details based on the requisition ID
            $items_data = [];
            $items_sql = "SELECT * FROM item_requisition_header_all WHERE req_id = ?";  
            $items_stmt = sqlsrv_query($conn, $items_sql, [$requisition_id]);

            while ($item_row = sqlsrv_fetch_array($items_stmt, SQLSRV_FETCH_ASSOC)) {
                // Loop through each item and its details
                for ($i = 1; $i <= 40; $i++) {
                    $item_id = $item_row["item_id_$i"];
                    $item_details_json = $item_row["item_details_$i"] ?? '';
                    $item_details = is_string($item_details_json) && $item_details_json !== ''
                        ? json_decode($item_details_json, true)
                        : [];

                    
                    if (!empty($item_id)) {
                        // Get item details from the `item_header_all` table
                        $item_detail_stmt = sqlsrv_query(
                            $conn,
                            "SELECT item_id, item_name, item_drawing_no, item_part_no FROM item_header_all WHERE item_id = ?",
                            [$item_id]
                        );
                        $item_info = sqlsrv_fetch_array($item_detail_stmt, SQLSRV_FETCH_ASSOC);

                        // Prepare the item details
                        if($item_details['status'] == 'A') {
                            $item_details_for_item = [
                                "item_id" => $item_id,
                                "item_name" => $item_info['item_name'],
                                "item_drawing_no" => $item_info['item_drawing_no'],
                                "item_part_no" => $item_info['item_part_no'],
                                "working_qty" => $item_details['working_qty'],
                                "rob" => $item_details['rob'],
                                "req_qty" => $item_details['req_qty'],
                                "accepted_qty" => $item_details['accepted_qty'],
                                "remark" => $item_details['remark']
                                // "status" => $item_details['status']
                            ];

                        // Add the item details to the items array
                        $items_data[] = [
                            "item_id" => $item_id,
                            "item_details" => $item_details_for_item
                        ];
                        }
                        
                    }
                }
            }

            $response = [
                'status' => 'success',
                'data' => [
                    'requisition_id' => $row['req_id'],  
                    'req_no' => $row['req_no'],
                    'creation_date' => $creation_date,
                    'priority_level' => $priority_level,
                    'ship_name' => $row['ship_name'],
                    'dept_name' => $row['dept_name'],
                    'req_serial_no' => $row['req_serial_no'],
                    'req_for' => $row['req_for'],
                    'req_eq_name_engine_type' => $row['req_eq_name_engine_type'],
                    'req_model_type' => $row['req_model_type'],
                    'req_maker_name' => $row['req_maker_name'],
                    'req_maker_address' => $row['req_maker_address'],
                    'req_remark' => $row['req_remark'],
                    'user_id' => $row['user_id'],
                    'req_view' => $row['req_view'],
                    'req_edit_count' => $row['req_edit_count'],
                    'req_edit_details' => $row['req_edit_details'],
                    'req_quotation_files' => $row['req_quotation_files'],
                    'items' => $items_data
                ]
            ];
        } else {
            $response = [
                'status' => 'error',
                'message' => 'No requisition found with the given requisition ID.'
            ];
        }

        echo json_encode($response);
    } else {
        // Pagination logic using req_id
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = 6;
        $offset = ($page - 1) * $limit;

        $sql = "SELECT r.req_id, r.req_no, r.ship_id, sha.ship_name, r.dept_id, dha.dept_name, r.req_status, r.req_date_of_creation
                FROM requisition_header_all r
                LEFT JOIN ship_header_all sha ON r.ship_id = sha.ship_id
                LEFT JOIN department_header_all dha ON r.dept_id = dha.dept_id
                WHERE r.req_status = 1
                ORDER BY r.req_id DESC OFFSET ? ROWS FETCH NEXT ? ROWS ONLY";  
        $stmt = sqlsrv_query($conn, $sql, [$offset, $limit]);

        $requests = [];
        while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
            $date = $row['req_date_of_creation'];
            $formattedDate = $date instanceof DateTime ? $date->format('m/d/Y') : '';

            $requests[] = [
                'requisition_id' => $row['req_id'],  
                'req_no' => $row['req_no'],  
                'ship_name' => $row['ship_name'],
                'department' => $row['dept_name'],
                'requested_date' => $formattedDate,
                'status' => $row['req_status']
            ];
        }

        $count_sql = "SELECT COUNT(*) AS total FROM requisition_header_all WHERE req_status = 1";
        $count_stmt = sqlsrv_query($conn, $count_sql);
        $count_row = sqlsrv_fetch_array($count_stmt, SQLSRV_FETCH_ASSOC);
        $total_records = $count_row['total'];
        $total_pages = ceil($total_records / $limit);

        $response = [
            'success_code' => 200,
            'message' => 'Requisitions fetched successfully',
            'requests' => $requests,
            'totalPages' => $total_pages
        ];

        echo json_encode($response);
    }
}
?>
