<?php

/*
objective : This API handles the creation and retrieval of users. It supports adding a new user and fetching users with pagination and optional search functionality.

file version : 1.0
API NO : KDN07
Date of creation : 28/03/2025

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Logic Zone : 

Date  : 28/03/2025

Request Method
Type: POST, GET

Headers: Content-Type: application/json

Request Parameters
- For POST:
  - name (string, required) – The full name of the user.
  - email (string, required) – The email address of the user.
  - mobile (string, required) – The mobile number of the user.
  - user_type (int, required) – The type of user (e.g., admin, regular).
  - profile (int, required) – The profile ID of the user.
  - username (string, required) – The username for the user to log in.
  - password (string, required) – The password for the user.

- For GET:
  - action (string, optional) – Specifies action type ('get_profiles' to fetch user profiles).
  - search (string, optional) – Search query for filtering users by name or email.
  - page (int, optional) – The page number for pagination. Default is 1.

Authentication: No authentication is required for this API.

***********

Authentication Process

* Step 1: Input Validation
  - For POST, it checks if name, email, mobile, user_type, profile, username, and password are provided.
  - For GET, it fetches user profiles or paginated user data based on the search query or the current page.

* Step 2: Sanitize Input
  - Ensures the provided email, mobile, and username are sanitized to prevent SQL injection.
  - Validates the email format and mobile number format.

* Step 3: Duplicate Check (POST only)
  - It checks if the provided email, username, or name already exists in the database.
  - If any of them exist, an error response is returned.

* Step 4: Inserting User (POST only)
  - If no duplicates are found, a new user is inserted into the database with the provided details, including user_no (which is generated based on the last user ID).

* Step 5: Fetching User Data (GET only)
  - For GET requests, it fetches a paginated list of users, optionally filtered by the search query.
  - It calculates the total number of pages for pagination based on the number of matching users.

***********

Response Structure

* Success Response
  - For POST:
    - success_code: 200 (indicating success)
    - message: 'User created successfully'

  - For GET:
    - success_code: 200 (indicating success)
    - data: Array of user details
    - total_pages: Total number of pages for pagination
    - current_page: The current page number

* Failure Responses:
  - For POST:
    - error_code: 100 (Email, Username, or Name already exists)
    - error_code: 101 (Invalid email format)
    - error_code: 102 (Invalid mobile number format or error adding user)
    - error_code: 400 (Missing required fields)

  - For GET:
    - error_code: 101 (No users found)
    - error_code: 300 (Invalid request method)

Date :
xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Error Zone : 

Date : 
*/


session_start();
require_once("../config/config.php");
$itemsPerPage = 5;
$conn = SqlServerConnection::getConnection();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $required = ['name', 'email', 'mobile', 'user_type', 'profile', 'username', 'password'];
    foreach ($required as $field) {
        if (empty($_POST[$field])) {
            echo json_encode(['error_code' => 400, 'message' => 'All fields are required']);
            exit;
        }
    }

    // Get & sanitize inputs
    $name = trim($_POST['name']);
    $email = trim($_POST['email']);
    $mobile = trim($_POST['mobile']);
    $user_type = (int)$_POST['user_type'];
    $profile_id = (int)$_POST['profile'];
    $username = trim($_POST['username']);
    $password = trim($_POST['password']);
    $user_date_of_creation = date('Y-m-d H:i:s');

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['error_code' => 101, 'message' => 'Invalid email format']);
        exit;
    }

    if (!preg_match("/^[6-9][0-9]{9}$/", $mobile)) {
        echo json_encode(['error_code' => 102, 'message' => 'Invalid mobile number']);
        exit;
    }

    // Duplicate check
    $check_sql = "SELECT * FROM user_header_all WHERE user_email = '$email' OR user_name_for_login = '$username' OR user_name = '$name'";
    $check_result = sqlsrv_query($conn, $check_sql);

    if (sqlsrv_has_rows($check_result)) {
        echo json_encode(['error_code' => 100, 'message' => 'Email, Username, or Name already exists']);
        exit;
    }

    $user_no = generateUserNo($conn);
    $insert_sql = "
        INSERT INTO user_header_all 
        (user_no, user_name, user_name_for_login, user_email, user_pass, user_mobile_no, user_type, user_profile_id, user_date_of_creation) 
        VALUES 
        ('$user_no', '$name', '$username', '$email', '$password', '$mobile', $user_type, $profile_id, '$user_date_of_creation')
    ";

    if (sqlsrv_query($conn, $insert_sql)) {
        echo json_encode(['success_code' => 200, 'message' => 'User created successfully']);
    } else {
        $errors = sqlsrv_errors();
        echo json_encode(['error_code' => 102, 'message' => 'Error adding user', 'errors' => $errors]);
    }
}
elseif ($_SERVER['REQUEST_METHOD'] === 'GET') {
    if (isset($_GET['action']) && $_GET['action'] === 'get_profiles') {
        $query = "SELECT profile_id, profile_name FROM profile_header_all";
        $result = sqlsrv_query($conn, $query);

        $profiles = [];
        while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
            $profiles[] = $row;
        }

        if (!empty($profiles)) {
            echo json_encode(['success_code' => 200, 'success' => true, 'profiles' => $profiles]);
        } else {
            echo json_encode(['error_code' => 101, 'message' => 'No profiles found']);
        }
        exit;
    }

    // Handle search & pagination
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    $currentPage = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($currentPage - 1) * $itemsPerPage;

    $searchCondition = "WHERE user_name LIKE '%$search%' OR user_email LIKE '%$search%'";

    $count_sql = "SELECT COUNT(*) AS total FROM user_header_all $searchCondition";
    $count_result = sqlsrv_query($conn, $count_sql);
    $totalUsers = sqlsrv_fetch_array($count_result, SQLSRV_FETCH_ASSOC)['total'];
    $totalPages = ceil($totalUsers / $itemsPerPage);

    $data_sql = "
        SELECT user_header_all.user_no, user_header_all.user_name, user_header_all.user_email, 
               user_header_all.user_mobile_no, user_header_all.user_type, profile_header_all.profile_name
        FROM user_header_all
        LEFT JOIN profile_header_all ON user_header_all.user_profile_id = profile_header_all.profile_id
        $searchCondition
        ORDER BY user_header_all.user_no
        OFFSET $offset ROWS FETCH NEXT $itemsPerPage ROWS ONLY
    ";

    $result = sqlsrv_query($conn, $data_sql);
    $users = [];

    while ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
        $users[] = $row;
    }

    if (!empty($users)) {
        echo json_encode([
            'success_code' => 200,
            'success' => true,
            'data' => $users,
            'total_pages' => $totalPages,
            'current_page' => $currentPage
        ]);
    } else {
        echo json_encode(['error_code' => 101, 'message' => 'No users found']);
    }
}
else {
    echo json_encode(['error_code' => 300, 'message' => 'Invalid request method']);
}

// Function to generate user_no
function generateUserNo($conn) {
    $prefix = 'user';
    $sql = "SELECT TOP 1 user_no FROM user_header_all ORDER BY user_no DESC";
    $result = sqlsrv_query($conn, $sql);

    if ($row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC)) {
        $last_no = $row['user_no'];
        $serial = (int)substr($last_no, 5);
        $new_serial = str_pad($serial + 1, 2, '0', STR_PAD_LEFT);
    } else {
        $new_serial = '01';
    }

    return "$prefix-$new_serial";
}
?>