<?php
/*
objective : This API handles the creation and retrieval of sub-categories within specific categories and departments. 
It supports adding new sub-categories to a category and department, and fetching sub-categories with pagination and optional search functionality.

file version : 1.0
API NO : KDN04
Date of creation : 27/03/2025

xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Logic Zone : 

Date  : 27/03/2025

Request Method
Type: POST, GET

Headers: Content-Type: application/json

Request Parameters
- For POST:
  - dept_id (int, required) – The ID of the department.
  - cat_id (int, required) – The ID of the category.
  - sub_cat_name (string, required) – The name of the sub-category.

- For GET:
  - action (string, optional) – Specifies action type ('fetch_departments' to fetch department details, 'fetch_categories' to fetch categories for a department).
  - dept_id (int, optional) – The department ID for which sub-categories are to be retrieved.
  - cat_id (int, optional) – The category ID for which sub-categories are to be retrieved.
  - search (string, optional) – Search query for filtering sub-category names.
  - page (int, optional) – The page number for pagination. Default is 1.

Authentication: No authentication is required for this API.

***********

Authentication Process

* Step 1: Input Validation
  - For POST, it checks if department ID, category ID, and sub-category name are provided.
  - For GET, it checks if department ID and category ID are provided (if applicable), and applies pagination with search.

* Step 2: Duplicate Check (POST only)
  - It checks if the sub-category already exists in the specified category.
  - If the sub-category exists, an error response is returned.

* Step 3: Inserting Sub-category (POST only)
  - If no duplicate is found, a new sub-category is inserted into the database with the provided department ID and category ID.
  
* Step 4: Fetching Sub-category Data (GET only)
  - For GET requests, it returns a paginated list of sub-categories, optionally filtered by department, category, and search query.
  - It calculates the total number of pages for pagination based on the number of matching sub-categories.

***********

Response Structure

* Success Response
  - For POST:
    - success_code: 200 (indicating success)
    - message: 'Sub-category added successfully'

  - For GET:
    - error_code: 200 (indicating success)
    - data: Array of sub-category details
    - total_pages: Total number of pages for pagination
    - current_page: The current page number

* Failure Responses:
  - For POST:
    - error_code: 100 (Sub-category already exists in this category)
    - error_code: 400 (Missing required department ID, category ID, or sub-category name)
    - error_code: 102 (Error while adding sub-category)

  - For GET:
    - error_code: 101 (No sub-categories found)
    - error_code: 300 (Invalid request method)

Date :
xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

Error Zone : 

Date : 
*/

session_start();
require_once("../config/config.php");

$itemsPerPage = 5;

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['dept_id']) && isset($_POST['cat_id']) && isset($_POST['sub_cat_name']) && !empty($_POST['dept_id']) && !empty($_POST['cat_id']) && !empty($_POST['sub_cat_name'])) {
        $conn = connection::getConnection();

        // Sanitize input to prevent SQL injection
        $dept_id = mysqli_real_escape_string($conn, $_POST['dept_id']);
        $cat_id = mysqli_real_escape_string($conn, $_POST['cat_id']);
        $sub_cat_name = mysqli_real_escape_string($conn, $_POST['sub_cat_name']);
        $sub_cat_status = 'N'; // Default status: Normal
        $sub_cat_date_of_creation = date('Y-m-d H:i:s');

        // Check if sub-category already exists for this category
        $checkSubCategoryQuery = "SELECT * FROM sub_category_header_all WHERE cat_id = '$cat_id' AND sub_cat_name = '$sub_cat_name'";
        $checkResult = mysqli_query($conn, $checkSubCategoryQuery);
        if (mysqli_num_rows($checkResult) > 0) {
            echo json_encode(['error_code' => 100, 'message' => 'Sub-category already exists in this category']);
            exit;
        }

        // Query to insert sub-category into the database
        $sql = "INSERT INTO sub_category_header_all (dept_id, cat_id, sub_cat_name, sub_cat_date_of_creation, sub_cat_status) 
                VALUES ('$dept_id', '$cat_id', '$sub_cat_name', '$sub_cat_date_of_creation', '$sub_cat_status')";

        if (mysqli_query($conn, $sql)) {
            echo json_encode(['success_code' => 200, 'message' => 'Sub-category added successfully']);
        } else {
            echo json_encode(['error_code' => 102, 'message' => 'Error adding sub-category: ' . mysqli_error($conn)]);
        }
    } else {
        echo json_encode(['error_code' => 400, 'message' => 'Department, Category, and Sub-category name are required']);
    }
} elseif ($_SERVER['REQUEST_METHOD'] == 'GET') {
    $conn = connection::getConnection();
    if (isset($_GET['action']) && $_GET['action'] == 'fetch_departments') {
        // Fetch departments
        $sql = "SELECT dept_id, dept_name FROM department_header_all";
        $result = mysqli_query($conn, $sql);

        $departments = [];
        while ($row = mysqli_fetch_assoc($result)) {
            $departments[] = $row;
        }
        echo json_encode($departments);
    } elseif (isset($_GET['action']) && $_GET['action'] == 'fetch_categories') {
        // Fetch categories for a specific department
        $dept_id = isset($_GET['dept_id']) ? (int)$_GET['dept_id'] : 0;
        $sql = "SELECT cat_id, cat_name FROM category_header_all WHERE dept_id = '$dept_id'";
        $result = mysqli_query($conn, $sql);

        $categories = [];
        while ($row = mysqli_fetch_assoc($result)) {
            $categories[] = $row;
        }
        echo json_encode($categories);
    } else {
        $dept_id = isset($_GET['dept_id']) ? (int)$_GET['dept_id'] : 0;
        $cat_id = isset($_GET['cat_id']) ? (int)$_GET['cat_id'] : 0;
        $search = isset($_GET['search']) ? mysqli_real_escape_string($conn, $_GET['search']) : '';
        $currentPage = isset($_GET['page']) ? (int)$_GET['page'] : 1;

        // Query to get total sub-category count for pagination with search
        if ($dept_id === 0 || $cat_id === 0) {
            $totalSubCategoriesQuery = "SELECT COUNT(*) AS total FROM sub_category_header_all WHERE sub_cat_name LIKE '%$search%'";
        } else {
            $totalSubCategoriesQuery = "SELECT COUNT(*) AS total FROM sub_category_header_all WHERE dept_id = '$dept_id' AND cat_id = '$cat_id' AND sub_cat_name LIKE '%$search%'";
        }
        $totalSubCategoriesResult = mysqli_query($conn, $totalSubCategoriesQuery);
        $totalSubCategories = mysqli_fetch_assoc($totalSubCategoriesResult)['total'];
        $totalPages = ceil($totalSubCategories / $itemsPerPage);
        $currentPage = max(1, min($currentPage, $totalPages));
        $start = ($currentPage - 1) * $itemsPerPage;

        // Query to fetch sub-categories for the current page with search
        if ($dept_id === 0 || $cat_id === 0) {
            $sql = "SELECT s.sub_cat_id, s.sub_cat_name, s.sub_cat_status, c.cat_name, d.dept_name 
                    FROM sub_category_header_all s 
                    JOIN category_header_all c ON s.cat_id = c.cat_id
                    JOIN department_header_all d ON s.dept_id = d.dept_id
                    WHERE s.sub_cat_name LIKE '%$search%' LIMIT $start, $itemsPerPage";
        } else {
            $sql = "SELECT s.sub_cat_id, s.sub_cat_name, s.sub_cat_status, c.cat_name, d.dept_name 
                    FROM sub_category_header_all s 
                    JOIN category_header_all c ON s.cat_id = c.cat_id
                    JOIN department_header_all d ON s.dept_id = d.dept_id
                    WHERE s.dept_id = '$dept_id' AND s.cat_id = '$cat_id' AND s.sub_cat_name LIKE '%$search%' LIMIT $start, $itemsPerPage";
        }

        $result = mysqli_query($conn, $sql);

        $subCategories = [];
        if ($result && mysqli_num_rows($result) > 0) {
            while ($row = mysqli_fetch_assoc($result)) {
                $subCategories[] = $row;
            }
            echo json_encode([
                'success_code' => 200,
                'success' => true,
                'data' => $subCategories,
                'total_pages' => $totalPages,
                'current_page' => $currentPage
            ]);
        } else {
            echo json_encode(['error_code' => 101, 'message' => 'No sub-categories found']);
        }
    }
} else {
    echo json_encode(['error_code' => 300, 'message' => 'Invalid request method']);
}
?>
