<?php
/*
    API: Save Requisition as Draft (KDN03)
    Version: 1.1
    Date: 2025-06-05
*/

error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();
require_once '../config/config.php';
header('Content-Type: application/json');

function escapeStr($str) {
    return str_replace("'", "''", $str);
}

function logError($label, $data) {
    file_put_contents(__DIR__ . '/debug_log.txt', "[$label] " . print_r($data, true) . "\n", FILE_APPEND);
}

// Check request method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['error_code' => 405, 'message' => 'Invalid request method. Please use POST with JSON body.']);
    exit;
}

// Read raw POST data and decode
$rawInput = file_get_contents("php://input");
logError('Raw POST Data', $rawInput);

$data = json_decode($rawInput, true);
if (!$data) {
    logError('Invalid JSON', $rawInput);
    echo json_encode(['error_code' => 400, 'message' => 'Invalid JSON input']);
    exit;
}

// Database connection
$conn = SqlServerConnection::getConnection();
if ($conn === false) {
    $errors = sqlsrv_errors();
    logError('SQL Connection Failed', $errors);
    echo json_encode(['error_code' => 500, 'message' => 'SQL Server connection failed', 'sql_errors' => $errors]);
    exit;
}

// Start transaction
sqlsrv_begin_transaction($conn);

try {
    $req_no = escapeStr($data['req_no'] ?? '');
    if (empty($req_no)) {
        throw new Exception('Missing req_no');
    }

    // Retrieve req_id
    $sql = "SELECT req_id FROM requisition_header_all WHERE req_no = '$req_no'";
    $stmt = sqlsrv_query($conn, $sql);
    if ($stmt && $row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $req_id = (int)$row['req_id'];
    } else {
        throw new Exception('Failed to retrieve req_id');
    }

    // Get current irq_line_no and current_col_no
    $query = "
        SELECT irq_line_no, current_col_no 
        FROM item_requisition_header_all 
        WHERE req_id = $req_id 
          AND irq_line_no = (
              SELECT MAX(irq_line_no) 
              FROM item_requisition_header_all 
              WHERE req_id = $req_id
          )";
    $result = sqlsrv_query($conn, $query);
    if (!$result) {
        throw new Exception('Failed to get current_col_no');
    }

    $row = sqlsrv_fetch_array($result, SQLSRV_FETCH_ASSOC);
    $irq_line_no = $row['irq_line_no'] ?? 1;
    $current_col_no = $row['current_col_no'] ?? 0;

    $items = $data['items'] ?? [];
    $max_items = 40;

    foreach ($items as $index => $item) {
        if ($index >= $max_items * 10) { // limit items to avoid infinite loop
            break;
        }

        $current_col_no++;
        if ($current_col_no > $max_items) {
            $irq_line_no++;
            $current_col_no = 1;
        }

        $item_id = (int)($item['item_id'] ?? 0);
        $cat_id = (int)($item['cat_id'] ?? 0);
        $sub_cat_id = (int)($item['sub_cat_id'] ?? 0);
        $sec_sub_cat_id = (int)($item['sec_sub_cat_id'] ?? 0);
        $third_sub_cat_id = (int)($item['third_sub_cat_id'] ?? 0);

        // Determine ref
        $req_for = ($data['req_for'] === 'Spare') ? 1 : (($data['req_for'] === 'Store') ? 2 : 0);
        $ref = 'NULL';
        if ($req_for === 1) {
            if ($third_sub_cat_id != 0) $ref = 4;
            elseif ($sec_sub_cat_id != 0) $ref = 3;
            elseif ($sub_cat_id != 0) $ref = 2;
            elseif ($cat_id != 0) $ref = 1;
        }

        // Build JSON details
        $details = [
            'req_no' => $req_no,
            'cat_id' => $cat_id,
            'sub_cat_id' => $sub_cat_id,
            'sec_sub_cat_id' => $sec_sub_cat_id,
            'third_sub_cat_id' => $third_sub_cat_id,
            'item_id' => $item_id,
            'drawing_no' => $item['drawing_no'] ?? '',
            'part_no' => $item['part_no'] ?? '',
            'working_qty' => $item['working_qty'] ?? '',
            'rob' => $item['rob'] ?? '',
            'req_qty' => $item['req_qty'] ?? '',
            'accepted_qty' => $item['req_qty'] ?? '',
            'unit' => $item['unit'] ?? '',
            'remark' => $item['remark'] ?? '',
            'status' => 'A'
        ];

        $jsonDetail = json_encode($details, JSON_UNESCAPED_UNICODE);
        if ($jsonDetail === false) {
            throw new Exception('JSON encoding failed: ' . json_last_error_msg());
        }

        if ($index == 0 && $current_col_no > 1) {
            // Update existing row
            $updateSQL = "
                UPDATE item_requisition_header_all 
                SET current_col_no = $current_col_no,
                    item_id_$current_col_no = $item_id,
                    ref_$current_col_no = '$ref',
                    item_details_$current_col_no = N'" . escapeStr($jsonDetail) . "'
                WHERE req_id = $req_id AND irq_line_no = $irq_line_no
            ";
            $updateResult = sqlsrv_query($conn, $updateSQL);
            if (!$updateResult) {
                throw new Exception('Update failed: ' . print_r(sqlsrv_errors(), true));
            }
        } else {
            // Insert new row
            $insertSQL = "
                INSERT INTO item_requisition_header_all 
                (req_id, irq_line_no, current_col_no, 
                item_id_$current_col_no, ref_$current_col_no, item_details_$current_col_no)
                VALUES 
                ($req_id, $irq_line_no, 1, 
                $item_id, '$ref', N'" . escapeStr($jsonDetail) . "')
            ";
            $insertResult = sqlsrv_query($conn, $insertSQL);
            if (!$insertResult) {
                throw new Exception('Insert failed: ' . print_r(sqlsrv_errors(), true));
            }
        }
    }

    // Commit
    sqlsrv_commit($conn);
    echo json_encode(['error_code' => 200, 'message' => 'Draft saved successfully']);

} catch (Exception $e) {
    sqlsrv_rollback($conn);
    logError('Exception', $e->getMessage());
    echo json_encode(['error_code' => 500, 'message' => $e->getMessage()]);
}

?>
